/* ---------------------------------------------------------------------------
                                cp1_tm1637.h

     Grundfunktionen des TM1637 7-Segment- und Keyboardcontrollers auf
     dem CP1+ Board

     21.05.2025    R. Seelig
   --------------------------------------------------------------------------- */

/*

    Segmentbelegung der Anzeige:

        a
       ---
    f | g | b            Segment |  a  |  b  |  c  |  d  |  e  |  f  |  g  |  dp |
       ---               ---------------------------------------------------------------------------------
    e |   | c            Bit-Nr. |  0  |  1  |  2  |  3  |  4  |  5  |  6  |  7  |
       ---
        d

*/

#include "tm16xx.h"

/* ----------------------------------------------------------
                     Globale Variable
   ---------------------------------------------------------- */

uint8_t    led7sbmp[16] =                // Bitmapmuster fuer Ziffern von 0 .. F
                { 0x3f, 0x06, 0x5b, 0x4f, 0x66, 0x6d, 0x7d, 0x07,
                  0x7f, 0x6f, 0x77, 0x7c, 0x39, 0x5e, 0x79, 0x71 };


/*  ------------------- Kommunikation -----------------------

    Der Treiberbaustein tm16 wird etwas "merkwuerdig
    angesprochen. Er verwendet zur Kommunikation ein I2C
    Protokoll, jedoch OHNE eine Adressvergabe. Der Chip ist
    somit IMMER angesprochen. Aus diesem Grund wird die
    Kommunikation mittels Bitbanging vorgenommen. Hierfuer
    kann jeder freie I/O Anschluss des Controllers verwendet
    werden (siehe defines am Anfang).

    Ausserdem erfolgt im Gegensatz zu vielen andern I2C Bau-
    steinen die serielle Ausgabe mit dem niederwertigsten Bit
    (LSB first)
   ---------------------------------------------------------- */

tm16xx::tm16xx(uint8_t scl, uint8_t sda, uint8_t CTRLTYPE)
{
  scl= scl;
  sda= sda;
  ctrl_type= CTRLTYPE;

  sda_init();
  scl_init();
  tm16_scl= scl;
  tm16_sda= sda;
}


void tm16xx::start(void)              // I2C Bus-Start
{

  bb_scl_hi();
  bb_sda_hi();
  puls_len();
  bb_sda_lo();
}

void tm16xx::stop(void)               // I2C Bus-Stop
{

  bb_scl_lo();
  puls_len();
  bb_sda_lo();
  puls_len();
  bb_scl_hi();
  puls_len();
  bb_sda_hi();

}

void tm16xx::write (uint8_t value)    // I2C Bus-Datentransfer
{
  uint8_t i;

  for (i = 0; i <8; i++)
  {

    bb_scl_lo();
    if (ctrl_type == TM1650_CTRL)
    {
      //  serielle Bitbangingausgabe, MSB first
      
      if (value & (uint8_t)0x80) { bb_sda_hi(); }
                            else { bb_sda_lo(); }
                     
      puls_len();
      value = value << 1;
      bb_scl_hi();
      puls_len();
    }
    else
    {
      //  serielle Bitbangingausgabe, LSB first
      if (value & 0x01) { bb_sda_hi(); }
                   else { bb_sda_lo(); }
      puls_len();
      value = value >> 1;
      bb_scl_hi();
      puls_len();
    }
  }
  bb_scl_lo();
  puls_len();                        // der Einfachheit wegen wird ACK nicht abgefragt
  bb_scl_hi();
  puls_len();
  bb_scl_lo();
}

/* -------------------------------------------------------
                    tm16xx::read(uint8_t ack)

   liest ein Byte vom I2c Bus.

   Uebergabe:
               1 : nach dem Lesen wird dem Slave ein
                   Acknowledge gesendet
               0 : es wird kein Acknowledge gesendet

   Rueckgabe:
               gelesenes Byte
   ------------------------------------------------------- */
uint8_t tm16xx::read(uint8_t ack)
{
  uint8_t data= 0x00;
  uint8_t i;
  uint8_t hin, lon;

  if (ctrl_type == TM1650_CTRL)
  {
    bb_sda_hi();

    for(i= 0; i< 8; i++)
    {

      bb_scl_lo();
      puls_len();

      if(bb_is_sda()) data|= (1 << i);

      bb_scl_hi();
      puls_len();

    }

    bb_scl_hi();
    puls_len();
    bb_scl_lo();
    puls_len();             // Zeit fuer den Acknowledge
    bb_sda_hi();

  //  return data;

    if (data & 1)
    {
      hin= ((data >> 5) & 0x3)+1;
      hin= (hin << 4);
      lon= ((data & 0x0f) >> 1)+1;
      data= hin | lon;
      return data;
    }
    else { return 0; }
  }
  else
  {
    bb_sda_hi();

    for(i= 0; i< 8; i++)
    {
      bb_scl_lo();
      puls_len();
      bb_scl_hi();

      puls_len();

      if(bb_is_sda()) data|= (1 << i);
    }

    bb_scl_lo();
    bb_sda_hi();

    puls_len();

    if (ack)
    {
      bb_sda_lo();
      puls_len();
    }

    bb_scl_hi();
    puls_len();

    bb_scl_lo();
    puls_len();

    bb_sda_hi();

    return data;
  }
}


/*  ----------------------------------------------------------
                      Benutzerfunktionen
    ---------------------------------------------------------- */
    
/* ---------------------------------------------------------
                        tm16xx::begin

        loescht die Anzeige und zeigt 0 an
   --------------------------------------------------------- */
void tm16xx::begin(void)
{
  setbright(1);
  clear();
  setdez6digit(0,0);
}

/* ---------------------------------------------------------
                        tm16xx::selectpos

        waehlt die zu beschreibende Anzeigeposition aus
   --------------------------------------------------------- */
void tm16xx::selectpos(int8_t nr)
{
  if (ctrl_type == TM1650_CTRL)
  {
    nr= 3-nr;
    start();
    write(0x40);                // Auswahl LED-Register
    stop();
    start();
    write(0x68 | (nr << 1));           // Auswahl der 7-Segmentanzeige
  }
  else
  {
    nr+= tm16_xofs;
    start();
    write(0x40);                // Auswahl LED-Register
    stop();
    start();

    write(0xc0 | nr);           // Auswahl der 7-Segmentanzeige
  }
}

/* ---------------------------------------------------------
                       tm16xx::setbright

       setzt die Helligkeit der Anzeige
       erlaubte Werte fuer Value sind 0 .. 7
       Wert 8 fuer Anzeige aus !
   ---------------------------------------------------------- */
void tm16xx::setbright(uint8_t value)
{
  if (ctrl_type == TM1650_CTRL)
  {
    hellig= (value % 8) << 4;
    start();
    write(0x48);
    write(0x01 | hellig);
    stop();
  }
  else
  {
    hellig= (value % 8) + 8;
    start();
    write(0x80 | hellig);        // unteres Nibble beinhaltet Helligkeitswert
    stop();
  }
}

/* ---------------------------------------------------------
                        tm16xx::clear

       loescht die Anzeige auf dem Modul
   --------------------------------------------------------- */
void tm16xx::clear()
{
  uint8_t i;

  if (ctrl_type == TM1650_CTRL)
  {

    for(i=0; i<4; i++)
    {
      selectpos(i);
      write(0x00);
    }
    stop();

    start();
    write(0x48);
    write(0x01 | hellig);
    stop();

  }
  else
  {
    selectpos(0);
    for(i=0; i<6; i++) { write(0x00); }
    stop();

    start();
    write(0x80 | hellig);        // unteres Nibble beinhaltet Helligkeitswert
    stop();
  }
}

/* ---------------------------------------------------------
                         tm16xx::setbmp

       gibt ein Bitmapmuster an einer Position aus
   --------------------------------------------------------- */
void tm16xx::setbmp(uint8_t pos, uint8_t value)
{
  selectpos(pos);             // zu beschreibende Anzeige waehlen

  write(value);               // Bitmuster value auf 7-Segmentanzeige ausgeben
  stop();

}

/* ---------------------------------------------------------
                         tm16xx::setzif

       gibt ein Ziffer an einer Position aus
       Anmerkung: das Bitmuster der Ziffern ist in
                  led7sbmp definiert
   --------------------------------------------------------- */
void tm16xx::setzif(uint8_t pos, uint8_t zif)
{
  selectpos(pos);             // zu beschreibende Anzeige waehlen

  zif= led7sbmp[zif];
  write(zif);               // Bitmuster value auf 7-Segmentanzeige ausgeben
  stop();

}

/* ---------------------------------------------------------
                          tm16xx::setseg

       setzt ein einzelnes Segment einer Anzeige

       pos: Anzeigeposition (0..3)
       seg: das einzelne Segment (0..7 siehe oben)
    --------------------------------------------------------- */
void tm16xx::setseg(uint8_t pos, uint8_t seg)
{

  selectpos(pos);             // zu beschreibende Anzeige waehlen
  write(1 << seg);
  stop();

}

/* ---------------------------------------------------------
                          tm16xx::setdez

       gibt einen 4-stelligen dezimalen Wert auf der
       Anzeige aus

       Uebergabe:

         value : auszugebender Wert
         dpanz : Position der Anzeige Dezimalpunkt, 1 ent-
                 spricht Anzeige rechts (und somit keine
                 Nachkommastelle)
                 0 => es erfolgt keine Anzeige
   --------------------------------------------------------- */
void tm16xx::setdez(int value, uint8_t dpanz)
{
  uint8_t i,v, bmp;
  uint8_t negflag;

  negflag= 0;
  if (value< 0)
  {
    value= -value;
    negflag++;
  }

  for (i= 4; i> 0; i--)
  {
    v= value % 10;
    bmp= led7sbmp[v];
    if (dpanz== 5-i) { bmp= bmp | 0x80; }     // Dezimalpunkt setzen
    if (!((i== 1) && negflag))
    {
      setbmp(i-1, bmp);
    }
    value= value / 10;
  }
  if (negflag) { setseg(0,6); }

}

/* ---------------------------------------------------------
                    tm16xx::setdez_nonull

     zeigt eine 4 stellige dezimale Zahl OHNE fuehrende
     Null an.

         Uebergabe:

           value : auszugebender Wert
           dpanz : Position der Anzeige Dezimalpunkt, 1 ent-
                   spricht Anzeige rechts (und somit keine
                   Nachkommastelle)
                   0 => es erfolgt keine Dezimalpunktanzeige
   --------------------------------------------------------- */
void tm16xx::setdez_nonull(int value, uint8_t dpanz)
{
  uint8_t  i,v, bmp, first;
  uint16_t teiler;
  uint8_t  negflag;

  teiler= 1000;
  first= 1;
  negflag= 0;
  if (value< 0)
  {
    value= -value;
    negflag++;
  }

  for (i= 1; i< 5; i++)
  {
    v= value / teiler;
    value= value - (v * teiler);
    teiler= teiler / 10;

    bmp= led7sbmp[v];
    if (dpanz== 5-i) bmp= bmp | 0x80;     // Dezimalpunkt setzen

    if (!((v== 0) && (first) && (i != 6)))
    {
      setbmp(i-1, bmp);
      first= 0;
    }
    else
    {
      // fuehrende Null nicht anzeigen
      if (!((i==1) && (negflag)))
      {
        if (dpanz== 5-i)
        {
          setbmp(i-1, 0x80);
        }
        else
        {
          setbmp(i-1, 0);
        }
      }
    }
  }
}


/* ----------------------------------------------------------
                          tm_setdez2

       gibt einen 2-stelligen dezimalen Wert auf der
       Anzeige aus

       pos:     0 => Anzeige erfolgt auf den hinteren
                     beiden Digits
                1 => Anzeige erfolgt auf den vorderen
                     beiden Digits
                2 => Anzeige erfolgt in der Mitte
   ---------------------------------------------------------- */
void tm16xx::setdez2(char pos, int8_t value)
{
  int8_t v;
  uint8_t  negflag;

  if (value< 0)
  {
    value= -value;
    negflag++;
    if (pos== 1) { pos= 2; }     // vorderes Digit fuer Minuszeichen reserviert
  }

  if (pos== 2)
  {
    v= value % 10;
    setbmp(2, led7sbmp[v]);
    value= value / 10;
    v= value % 10;
    setbmp(1, led7sbmp[v]);
  }
  else
  {
    pos= pos % 2;
    pos= (1 - pos) * 2;
    v= value % 10;
    setbmp(pos+1, led7sbmp[v]);
    value= value / 10;
    v= value % 10;
    setbmp(pos, led7sbmp[v]);
  }
  if (negflag) { setseg(0,6); }
}


/* ---------------------------------------------------------
                       tm16xx::setdez6digit

       gibt einen 6-stelligen dezimalen Wert auf der
       Anzeige aus

       Uebergabe:

         value : auszugebender Wert
         dpanz : Position der Anzeige Dezimalpunkt, 1 ent-
                 spricht Anzeige rechts (und somit keine
                 Nachkommastelle)
                 0 => es erfolgt keine Anzeige
   --------------------------------------------------------- */
void tm16xx::setdez6digit(uint32_t value, uint8_t dpanz)
{
  uint8_t i,v, bmp;

  for (i= 6; i> 0; i--)
  {
    v= value % 10;
    bmp= led7sbmp[v];
    if (dpanz== 7-i) bmp= bmp | 0x80;     // Dezimalpunkt setzen
    setbmp(i-1, bmp);
    value= value / 10;
  }
}


/* ---------------------------------------------------------
                    tm16xx::setdez6digit_nonull

     zeigt eine max. 6 stellige dezimale Zahl OHNE fuehrende
     Null an.

         Uebergabe:

           value : auszugebender Wert
           dpanz : Position der Anzeige Dezimalpunkt, 1 ent-
                   spricht Anzeige rechts (und somit keine
                   Nachkommastelle)
                   0 => es erfolgt keine Dezimalpunktanzeige
   --------------------------------------------------------- */
void tm16xx::setdez6digit_nonull(int32_t value, uint8_t dpanz)
{
  uint8_t  i,v, bmp, first;
  uint32_t teiler;
  uint8_t  negflag;

  teiler= 100000;
  first= 1;
  negflag= 0;
  if (value< 0)
  {
    value= -value;
    setseg(0,6);
    negflag++;
  }

  for (i= 1; i< 7; i++)
  {
    v= value / teiler;
    value= value - (v * teiler);
    teiler= teiler / 10;

    bmp= led7sbmp[v];
    if (dpanz== 7-i) bmp= bmp | 0x80;     // Dezimalpunkt setzen

    if (!((v== 0) && (first) && (i != 6)))
    {
      setbmp(i-1, bmp);
      first= 0;
    }
    else
    {
      // fuehrende Null nicht anzeigen
      if (!((i==1) && (negflag)))
      {
        if (dpanz== 7-i)
        {
          setbmp(i-1, 0x80);
        }
        else
        {
          setbmp(i-1, 0);
        }
      }
    }
  }
}

/* ---------------------------------------------------------
                          tm16xx::sethex
       gibt einen 4-stelligen hexadezimalen Wert auf der
       Anzeige aus
   --------------------------------------------------------- */
void tm16xx::sethex(uint16_t value)
{
  uint8_t i,v;

  for (i= 4; i> 0; i--)
  {
    v= value % 0x10;
    setbmp(i-1, led7sbmp[v]);
    value= value / 0x10;
  }
}

/* ---------------------------------------------------------
                          tm_sethex2

       gibt einen 2-stelligen hexadezimalen Wert auf der
       Anzeige aus

       pos:     0 => Anzeige erfolgt auf den hinteren
                     beiden Digits
                1 => Anzeige erfolgt auf den vorderen
                     beiden Digits
    --------------------------------------------------------- */
void tm16xx::sethex2(char pos, uint8_t value)
{
  uint8_t v;

  pos= pos % 2;
  pos= (1 - pos) * 2;
  v= value & 0x0f;
  setbmp(pos+1, led7sbmp[v]);
  v= (value >> 4) & 0x0f;
  setbmp(pos, led7sbmp[v]);
}

/* ---------------------------------------------------------
                       tm16xx::sethex6digit

       gibt einen 6-stelligen hexadezimalen Wert auf der
       Anzeige aus
    --------------------------------------------------------- */
void tm16xx::sethex6digit(uint32_t value)
{
  uint8_t i,v;

  for (i= 6; i> 0; i--)
  {
    v= value % 0x10;
    setbmp(i-1, led7sbmp[v]);
    value= value / 0x10;
  }
}

/* ---------------------------------------------------------
                          tm16xx::readkey

      liest angeschlossene Tasten ein und gibt dieses als
      Argument zurueck.

      Anmerkung:
        Es wird keine Tastenmatrix zurueck geliefert. Ist
        mehr als eine Taste aktiviert, wird nur die hoechste
        Taste zurueck geliefert. Somit ist es nicht moeglich
        mehrere Tasten gleichzeitig zu betaetigen.

   --------------------------------------------------------- */
uint8_t tm16xx::readkey(void)
{
  uint8_t data;
  uint8_t i;
  uint8_t hin, lon;
  uint8_t key;

  if (ctrl_type == TM1650_CTRL)
  {

    data= 0;

    bb_sda_hi();

    for(i= 0; i< 8; i++)
    {

      bb_scl_lo();
      puls_len();

      if(bb_is_sda()) data|= (1 << i);

      bb_scl_hi();
      puls_len();

    }

    bb_scl_hi();
    puls_len();
    bb_scl_lo();
    puls_len();             // Zeit fuer den Acknowledge
    bb_sda_hi();

  //  return data;

    if (data & 1)
    {
      hin= ((data >> 5) & 0x3)+1;
      hin= (hin << 4);
      lon= ((data & 0x0f) >> 1)+1;
      data= hin | lon;
      return data;
    }
    else { return 0; }
  }
  else
  {
    key= 0;
    start();
    write(0x42);
    key= ~read(1);
    stop();
    if (key) key -= 7;
    return key;
  }
}
